"use client"

import { zodResolver } from "@hookform/resolvers/zod"
import { useForm } from "react-hook-form"
import { z } from "zod"
import { Form, FormControl, FormField, FormItem, FormLabel, FormMessage } from "@/components/ui/form"
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group"
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select"
import { Label } from "@/components/ui/label"
import { Card } from "@/components/ui/card"

const formSchema = z.object({
  role: z.string().min(1, "Please select your role"),
  useCase: z.string().min(1, "Please select your primary use case"),
})

interface RoleSelectionStepProps {
  userData: {
    role: string
    useCase: string
    [key: string]: string
  }
  updateUserData: (data: Partial<RoleSelectionStepProps["userData"]>) => void
}

export function RoleSelectionStep({ userData, updateUserData }: RoleSelectionStepProps) {
  const form = useForm<z.infer<typeof formSchema>>({
    resolver: zodResolver(formSchema),
    defaultValues: {
      role: userData.role,
      useCase: userData.useCase,
    },
  })

  const onSubmit = (values: z.infer<typeof formSchema>) => {
    updateUserData(values)
  }

  // Auto-save as user selects
  form.watch((data) => {
    if (Object.values(form.formState.errors).length === 0) {
      updateUserData(data as Partial<RoleSelectionStepProps["userData"]>)
    }
  })

  return (
    <div className="space-y-6 py-4">
      <div className="space-y-2">
        <h2 className="text-xl font-semibold">Your Role & Goals</h2>
        <p className="text-muted-foreground">
          Help us understand how you'll be using our product so we can optimize your experience.
        </p>
      </div>

      <Form {...form}>
        <form onSubmit={form.handleSubmit(onSubmit)} className="space-y-6">
          <FormField
            control={form.control}
            name="role"
            render={({ field }) => (
              <FormItem className="space-y-3">
                <FormLabel>What's your role?</FormLabel>
                <FormControl>
                  <RadioGroup
                    onValueChange={field.onChange}
                    defaultValue={field.value}
                    className="grid grid-cols-2 gap-4"
                  >
                    {[
                      { value: "executive", label: "Executive/C-Level" },
                      { value: "manager", label: "Manager/Team Lead" },
                      { value: "individual", label: "Individual Contributor" },
                      { value: "other", label: "Other" },
                    ].map((role) => (
                      <Card key={role.value} className="relative">
                        <Label
                          htmlFor={`role-${role.value}`}
                          className="flex items-center space-x-2 p-4 cursor-pointer"
                        >
                          <RadioGroupItem
                            value={role.value}
                            id={`role-${role.value}`}
                            className="absolute top-4 right-4"
                          />
                          <div>
                            <p className="font-medium">{role.label}</p>
                          </div>
                        </Label>
                      </Card>
                    ))}
                  </RadioGroup>
                </FormControl>
                <FormMessage />
              </FormItem>
            )}
          />

          <FormField
            control={form.control}
            name="useCase"
            render={({ field }) => (
              <FormItem>
                <FormLabel>What's your primary use case?</FormLabel>
                <Select onValueChange={field.onChange} defaultValue={field.value}>
                  <FormControl>
                    <SelectTrigger>
                      <SelectValue placeholder="Select your primary use case" />
                    </SelectTrigger>
                  </FormControl>
                  <SelectContent>
                    <SelectItem value="productivity">Improving team productivity</SelectItem>
                    <SelectItem value="collaboration">Enhancing collaboration</SelectItem>
                    <SelectItem value="automation">Automating workflows</SelectItem>
                    <SelectItem value="analytics">Data analysis and reporting</SelectItem>
                    <SelectItem value="customer">Customer relationship management</SelectItem>
                    <SelectItem value="other">Other</SelectItem>
                  </SelectContent>
                </Select>
                <FormMessage />
              </FormItem>
            )}
          />
        </form>
      </Form>
    </div>
  )
}
