"use client"

import { useState, type ReactNode } from "react"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardFooter, CardHeader } from "@/components/ui/card"
import { Progress } from "@/components/ui/progress"
import { ArrowLeft, ArrowRight, Check } from "lucide-react"

interface Step {
  id: string
  title: string
  component: ReactNode
}

interface OnboardingWizardProps {
  steps: Step[]
  onComplete: () => void
}

export function OnboardingWizard({ steps, onComplete }: OnboardingWizardProps) {
  const [currentStepIndex, setCurrentStepIndex] = useState(0)
  const currentStep = steps[currentStepIndex]
  const isFirstStep = currentStepIndex === 0
  const isLastStep = currentStepIndex === steps.length - 1

  const progress = (currentStepIndex / (steps.length - 1)) * 100

  const handleNext = () => {
    if (isLastStep) {
      onComplete()
    } else {
      setCurrentStepIndex((prev) => prev + 1)
    }
  }

  const handlePrevious = () => {
    setCurrentStepIndex((prev) => prev - 1)
  }

  return (
    <div className="space-y-6">
      <div className="space-y-2">
        <div className="flex items-center justify-between">
          <h2 className="text-sm font-medium text-gray-500">
            Step {currentStepIndex + 1} of {steps.length}
          </h2>
          <span className="text-sm font-medium text-gray-500">{Math.round(progress)}% Complete</span>
        </div>
        <Progress value={progress} className="h-2" />
      </div>

      <div className="flex items-center justify-center space-x-2">
        {steps.map((step, index) => (
          <div key={step.id} className="flex items-center">
            <div
              className={`flex h-8 w-8 items-center justify-center rounded-full border ${
                index <= currentStepIndex
                  ? "border-primary bg-primary text-primary-foreground"
                  : "border-gray-300 bg-white text-gray-500"
              }`}
            >
              {index < currentStepIndex ? <Check className="h-4 w-4" /> : <span>{index + 1}</span>}
            </div>
            {index < steps.length - 1 && (
              <div className={`h-1 w-10 ${index < currentStepIndex ? "bg-primary" : "bg-gray-200"}`} />
            )}
          </div>
        ))}
      </div>

      <Card className="shadow-lg">
        <CardHeader>
          <h1 className="text-2xl font-bold">{currentStep.title}</h1>
        </CardHeader>
        <CardContent>{currentStep.component}</CardContent>
        <CardFooter className="flex justify-between border-t p-6">
          <Button
            variant="outline"
            onClick={handlePrevious}
            disabled={isFirstStep}
            className={isFirstStep ? "invisible" : ""}
          >
            <ArrowLeft className="mr-2 h-4 w-4" />
            Back
          </Button>
          <Button onClick={handleNext}>
            {isLastStep ? (
              "Complete"
            ) : (
              <>
                Next
                <ArrowRight className="ml-2 h-4 w-4" />
              </>
            )}
          </Button>
        </CardFooter>
      </Card>
    </div>
  )
}
